
********************************************************************************
*                                                                              *
* 2 by 4 between subject design: Examples 5 to 7                               *
*                                                                              *
********************************************************************************

clear
set more off

use between2by4data 

browse

**********************
* Conventional ANOVA *
**********************

// Let us first run the traditional ANOVA to obtain the SSmodel:

anova retention sunglasses##location

// SSmodel is stored as e(mss) in Stata's memory. Let us store it in Mata's 
// memory for calculating ETA^2 alerting later:

mata
SSmodel = st_numscalar("e(mss)")
end


******************************************
* Example 5: A direct test of the theory *
******************************************

// As explained in the manuscript, we typically can ignore the factorial
// structure of the experimental design, and treat the 2 by 4 design as a 1  
// by 8. For this purpose, we need to make a new variable named "groups" that 
// identifies these 8 groups:

egen groups = group(sunglasses location)

// By treating the design as a 1 by 8 design, the remainder of the analysis is
// the same as for oneway between-subject designs (Example 1 to 4). First, we 
// run the MANOVA with retention as the dependent variable and the newly created 
// groups varianle as factor:

manova retention = groups

// Next we define the required row vector L with contrast weights, including the 
// weight of the intercept of grand mean in the last column (see annotated Stata
// syntax of Example 1):

matrix L = (3, 1, -1, -3/5, -3/5, -3/5, -3/5, -3/5, 0)

// We use this vector L as input to the test option of the manovatest 
// postestimation command:

manovatest, test(L)

// As in Examples 1 to 4, the SScontrast is stored in Stata's memory in a matrix
// called "r(H)". Let us store this in Mata's memory, so that it is available 
// later on for calculating ETA^2 alerting:

mata
SScontrast = st_matrix("r(H)")
SScontrast
end

// Finally, a similar syntax as in Examples 1 to 4 can be used to obtain 
// SScontrast, the C-estimate, and ETA^2 alerting. The only difference is that
// we now have 8 groups rather than 4, and thus that 8 constrast weights are
// are used in the calculation of the C-estimate. As before we start with 
// obtaining the emperical or observed group means:

mean retention, over(groups)

mata
Means = st_matrix("e(b)")
L = st_matrix("L") 
Cest = rowsum(L[1,1::(cols(L) - 1)] :* Means) 	  
Cest	
Eta2alerting = SScontrast/SSmodel
Eta2alerting				
end


*********************************************************
* Example 6: Negative linear effect of seating location *
* without but not with sunglasses?                      *
*********************************************************

// In contrast to SPSS, Stata cannot test two separate theorectical predictions 
// in one run. Instead the analysis has to be repeated for each contrast 
// separately. The required syntax is the same as in Example 5, except for the 
// constrast weights used. Note also that the "groups" variable needed to 
// analyse the data as coming from 1 by 8 design, the SSmodel, and the observed
// group means (vector "Means") are already available from the analysis of 
// Example 5. 

// First we define to vectors L1 and L2 with contrast weights: one for each 
// theorectical prediction:

matrix L1 = (3, 1, -1, -3, 0, 0, 0, 0, 0)
matrix L2 = (0, 0, 0, 0, 3, 1, -1, -3, 0)

// Let us first test against the emperical data the prediction defined in L1:

manova retention = groups
manovatest, test(L1)

mata
SScontrast = st_matrix("r(H)")
SScontrast
L = st_matrix("L1") 
Cest = rowsum(L[1,1::(cols(L) - 1)] :* Means) 	  
Cest
Eta2alerting = SScontrast/SSmodel
Eta2alerting				
end

// Finally, we test L2 against the emperical data:

manova retention = groups
manovatest, test(L2)

mata
SScontrast = st_matrix("r(H)")
SScontrast
L = st_matrix("L2") 
Cest = rowsum(L[1,1::(cols(L) - 1)] :* Means) 	  
Cest	
Eta2alerting = SScontrast/SSmodel
Eta2alerting				
end

****************************************************
* Example 7: Negative linear trend more pronounced *
* without than with sunglasses?                    *
****************************************************

// Testing the theorectical prediction described in Example 7 requires the same
// syntax as used in Examples 5 and 6, except of course for contrast weights  
// used. Note also that the "groups" variable needed to analyse the data as 
// coming from 1 by 8 design, the SSmodel, and the observed group means 
// (vector "Means") are already available from the analysis of Example 5:

matrix L = (3, 1, -1, -3, -3, -1, 1, 3, 0)
manova retention = groups
manovatest, test(L)

mata
SScontrast = st_matrix("r(H)")
SScontrast
L = st_matrix("L") 
Cest = rowsum(L[1,1::(cols(L) - 1)] :* Means) 	  
Cest	
Eta2alerting = SScontrast/SSmodel
Eta2alerting				
end

